<?php
/* This file has been prefixed by <PHP-Prefixer> for "XT Adaptive Images" */

/*
 * @package     XT Adaptive Images
 *
 * @author      Extly, CB. <team@extly.com>
 * @copyright   Copyright (c)2012-2022 Extly, CB. All rights reserved.
 * @license     https://www.gnu.org/licenses/gpl-3.0.html GNU/GPL
 *
 * @see         https://www.extly.com
 */

namespace XTP_BUILD\Extly\AdaptiveImages\Common;

use XTP_BUILD\Extly\Infrastructure\Creator\CreatorTrait;

class BrowserFeaturesHelper
{
    use CreatorTrait;

    public function getResolution($resolutions)
    {
        $resolution = $this->getCookieResolution($resolutions);

        // No resolution was found (no cookie or invalid cookie)
        if (!$resolution) {
            // Does the UA string indicate this is a mobile?
            // We send the lowest resolution for mobile-first approach, and highest otherwise
            $resolution = $this->isMobile() ? min($resolutions) : max($resolutions);
        }

        $this->resolution = $resolution;

        return $resolution;
    }

    /**
     * isMobile - Mobile detection NOTE: only used if the cookie isn't available.
     *
     * @return bool
     */
    private function isMobile()
    {
        if (\array_key_exists('HTTP_USER_AGENT', $_SERVER)) {
            $userAgent = strtolower($_SERVER['HTTP_USER_AGENT']);

            return false !== strpos($userAgent, 'mobile');
        }

        return false;
    }

    private function getCookieResolution($resolutions)
    {
        // Check to see if a valid cookie exists
        if (!isset($_COOKIE['resolution'])) {
            return null;
        }

        $cookieValue = $_COOKIE['resolution'];

        // Does the cookie look valid? [whole number, comma, potential floating number]
        if (!preg_match('/^[0-9]+[,]*[0-9\\.]+$/', "{$cookieValue}")) {
            // No it doesn't look valid
            setcookie('resolution', "{$cookieValue}", time() - 100); // delete the mangled cookie

            return null;
        }

        // The cookie is valid, do stuff with it
        $cookieData = explode(',', $_COOKIE['resolution']);

        // The base resolution (CSS pixels)
        $clientWidth = (int) $cookieData[0];
        $totalWidth = $clientWidth;

        // Set a default, used for non-retina style JS snippet
        $pixelDensity = 1;

        if (@$cookieData[1]) {
            // The device's pixel density factor (physical pixels per CSS pixel)
            $pixelDensity = $cookieData[1];
        }

        // Make sure the supplied break-points are in reverse size order
        rsort($resolutions);

        // By default use the largest supported break-point
        $resolution = $resolutions[0];

        // If pixel density is not 1, then we need to be smart about adapting and fitting into the defined breakpoints
        if (1 !== $pixelDensity) {
            // Required physical pixel width of the image
            $totalWidth = $clientWidth * $pixelDensity;

            // The required image width is bigger than any existing value in $resolutions
            if ($totalWidth > $resolutions[0]) {
                // Firstly, fit the CSS size into a break point ignoring the multiplier
                foreach ($resolutions as $breakPoint) {
                    // Filter down
                    if ($totalWidth <= $breakPoint) {
                        $resolution = $breakPoint;
                    }
                }

                // Now apply the multiplier
                $resolution = $resolution * $pixelDensity;
            }

            // The required image fits into the existing breakpoints in $resolutions
            else {
                foreach ($resolutions as $breakPoint) {
                    // Filter down
                    if ($totalWidth > $breakPoint) {
                        break;
                    }

                    $resolution = $breakPoint;
                }
            }
        } else {
            // Pixel density is 1, just fit it into one of the breakpoints
            foreach ($resolutions as $breakPoint) {
                // Filter down
                if ($totalWidth > $breakPoint) {
                    break;
                }

                $resolution = $breakPoint;
            }
        }

        return $resolution;
    }
}
