<?php
/* This file has been prefixed by <PHP-Prefixer> for "XT Adaptive Images" */

/*
 * @package     XT Adaptive Images
 *
 * @author      Extly, CB. <team@extly.com>
 * @copyright   Copyright (c)2012-2022 Extly, CB. All rights reserved.
 * @license     https://www.gnu.org/licenses/gpl-3.0.html GNU/GPL
 *
 * @see         https://www.extly.com
 */

namespace XTP_BUILD\Extly\AdaptiveImages\Common;

/*
 * ===================================================
 *  Inspired on Adaptive Images by Matt Wilcox http://adaptive-images.com
 * ===================================================
 * Homepage:  http://adaptive-images.com
 * GitHub:    https://github.com/MattWilcox/Adaptive-Images
 * Twitter:   @responsiveimg
 * ===================================================
 */

/**
 * AdaptiveImagesForJoomlaHelper class - Plugin that replaces media urls with Adaptive Images urls.
 *
 * @since       1.0
 */
class ImageProcessor
{
    const PROCESSING_LIMIT = 16;

    private $documentRoot;

    private $aiCachePath;

    private $aiResolutions;

    private $aiQuality;

    private $aiSharpen;

    private $aiConvertWebp;

    private $generateSrcset = false;

    private $srcSet = [];

    private $cacheResolution = true;

    private $resolution;

    private $numberOfProcessedImages;

    /**
     * ImageProcessor.
     *
     * @param string $documentRoot   Param
     * @param string $aiCachePath    Param
     * @param string $resolutions    Param
     * @param string $generateSrcset Param
     * @param mixed  $quality
     * @param mixed  $sharpen
     * @param mixed  $convertWebp
     */
    public function __construct($documentRoot, $aiCachePath, $resolutions, $quality, $sharpen, $convertWebp = false)
    {
        $this->documentRoot = $documentRoot;
        $this->aiCachePath = $aiCachePath;
        $this->aiResolutions = $resolutions;
        $this->aiQuality = $quality;
        $this->aiSharpen = $sharpen;
        $this->aiConvertWebp = $convertWebp;
        $this->numberOfProcessedImages = 0;

        // Does the $cachePath directory exist already?
        $cachePath = $this->documentRoot.'/'.$this->aiCachePath;

        if (!is_dir($cachePath)) {
            // No
            if (!mkdir($cachePath, 0755, true)) {
                // So make it
                if (!is_dir($cachePath)) {
                    // Check again to protect against race conditions
                    // Uh-oh, failed to make that directory
                    throw new \Exception('Failed to create cache directory at: '.$cachePath);
                }
            }
        }
    }

    /**
     * Get the value of generateSrcset.
     */
    public function getGenerateSrcset()
    {
        return $this->generateSrcset;
    }

    /**
     * Set the value of generateSrcset.
     *
     * @param mixed $generateSrcset
     *
     * @return self
     */
    public function setGenerateSrcset($generateSrcset)
    {
        $this->generateSrcset = $generateSrcset;

        return $this;
    }

    /**
     * Get the value of srcSet.
     */
    public function getSrcSet()
    {
        $resolutions = $this->srcSet;
        ksort($resolutions);

        return $resolutions;
    }

    /**
     * Set the value of srcSet.
     *
     * @param mixed $srcSet
     *
     * @return self
     */
    public function setSrcSet($srcSet)
    {
        $this->srcSet = $srcSet;

        return $this;
    }

    /**
     * Get the value of cacheResolution.
     */
    public function getCacheResolution()
    {
        return $this->cacheResolution;
    }

    /**
     * Set the value of cacheResolution.
     *
     * @param mixed $cacheResolution
     *
     * @return self
     */
    public function setCacheResolution($cacheResolution)
    {
        $this->cacheResolution = $cacheResolution;

        return $this;
    }

    public function isSvg($file)
    {
        return '.svg' === substr($file, -4);
    }

    public function setConvertWebP($convertWebp)
    {
        $this->aiConvertWebp = $convertWebp;
    }

    /**
     * getAdaptedImage.
     *
     * @param string $file Params
     *
     * @return string
     */
    public function getAdaptedImage($file)
    {
        $this->srcSet = [];
        $sourceFile = $this->documentRoot.'/'.$file;

        // Check if the file exists
        if (!file_exists($sourceFile)) {
            return false;
        }

        // Is it a SVG?
        if ($this->isSvg($file)) {
            return $file;
        }

        // Check that PHP has the GD library available to use for image re-sizing
        if (!\extension_loaded('gd')) {
            // And we can't load it either - No GD available
            throw new \Exception('You must enable the GD extension to make use of XT Adaptive Images');
        }

        // The resolution break-points to use (screen widths, in pixels)
        $resolutions = explode(',', $this->aiResolutions);

        $requestedUri = '/'.$file;

        // If the requested URL starts with a slash, remove the slash
        if ('/' === substr($requestedUri, 0, 1)) {
            $requestedUri = substr($requestedUri, 1);
        }

        $sendImage = $this->generateImage($sourceFile, $requestedUri, $this->getResolution($resolutions));

        if (!$this->generateSrcset) {
            // Nothing else to do
            return $sendImage;
        }

        foreach ($resolutions as $resolution) {
            $file = $this->generateImage($sourceFile, $requestedUri, $resolution);

            // We have a new file
            if ($file !== $sourceFile) {
                $relativeFile = str_replace($this->documentRoot.'/', '', $file);
                $this->srcSet[$resolution] = $relativeFile;
            }
        }

        if (!empty($this->srcSet)) {
            $dimensions = getimagesize($sourceFile);
            $width = $dimensions[0];
            $relativeFile = str_replace($this->documentRoot.'/', '', $sourceFile);
            $this->srcSet[$width] = $relativeFile;
        }

        return $sendImage;
    }

    /**
     * generateImage - generates the given cache file for the given source file with the given resolution.
     *
     * @param string $sourceFile   Params
     * @param int    $requestedUri Params
     * @param int    $resolution   Params
     *
     * @return string
     */
    public function generateImage($sourceFile, $requestedUri, $resolution)
    {
        return $this->generateProcessedImage($sourceFile, $requestedUri, $resolution);
    }

    private function generateProcessedImage($sourceFile, $requestedUri, $resolution)
    {
        if (\defined('TESTING')) {
            return $this->processCachedFileOrGenerate($sourceFile, $requestedUri, $resolution);
        }

        static $processed = [];
        $key = md5($sourceFile.$resolution);

        if (isset($processed[$key])) {
            return $processed[$key];
        }

        $adaptedImage = $this->processCachedFileOrGenerate($sourceFile, $requestedUri, $resolution);
        $processed[$key] = $adaptedImage;

        return $adaptedImage;
    }

    private function processCachedFileOrGenerate($sourceFile, $requestedUri, $resolution)
    {
        $cacheFile = $this->documentRoot.'/'.$this->aiCachePath.'/'.$resolution.'/'.$requestedUri;
        $optimizedCacheFile = $this->documentRoot.'/'.$this->aiCachePath.'/optimized/'.$resolution.'/'.$requestedUri;

        if ($this->aiConvertWebp) {
            $webpCacheFile = $this->webpFilename($cacheFile);
            $webpOptimizedCacheFile = $this->webpFilename($optimizedCacheFile);

            return $this->checkCachedFileOrGenerate($sourceFile, $webpCacheFile, $webpOptimizedCacheFile, $resolution);
        }

        return $this->checkCachedFileOrGenerate($sourceFile, $cacheFile, $optimizedCacheFile, $resolution);
    }

    private function checkCachedFileOrGenerate($sourceFile, $cacheFile, $optimizedCacheFile, $resolution)
    {
        $fileCacheHelper = FileCacheHelper::create();

        // Use the resolution value as a path variable and check to see if an image of the same name exists at that path
        if (file_exists($optimizedCacheFile)) {
            // If cache watching is enabled, compare cache and source modified dates to ensure the cache isn't stale
            if ($fileCacheHelper->isCacheFresh($sourceFile, $optimizedCacheFile) &&
                $fileCacheHelper->isCacheFresh($sourceFile, $cacheFile)) {
                return $optimizedCacheFile;
            }
        }

        if (file_exists($cacheFile)) {
            // If cache watching is enabled, compare cache and source modified dates to ensure the cache isn't stale
            if ($fileCacheHelper->isCacheFresh($sourceFile, $cacheFile)) {
                return $cacheFile;
            }
        }

        return $this->generateCacheFile($sourceFile, $resolution, $cacheFile);
    }

    private function generateCacheFile($sourceFile, $resolution, $cacheFile)
    {
        $extension = strtolower(pathinfo($sourceFile, \PATHINFO_EXTENSION));

        // Check the image dimensions
        $dimensions = getimagesize($sourceFile);
        $width = $dimensions[0];
        $height = $dimensions[1];

        // Do we need to downscale the image?
        if ($width <= $resolution) {
            // No, because the width of the source image is already less than the client width
            return $sourceFile;
        }

        if ($this->numberOfProcessedImages >= self::PROCESSING_LIMIT) {
            // No, because it exhausted the processing limit
            return $sourceFile;
        }

        ++$this->numberOfProcessedImages;

        // We need to resize the source image to the width of the resolution breakpoint we're working with
        $ratio = $height / $width;
        $newWidth = $resolution;
        $newHeight = ceil($newWidth * $ratio);

        // Re-sized image
        $dst = imagecreatetruecolor($newWidth, $newHeight);

        switch ($extension) {
            case 'png':
                // Original image
                $src = @imagecreatefrompng($sourceFile);

                break;
            case 'gif':
                // Original image
                $src = @imagecreatefromgif($sourceFile);

                break;
            case 'webp':
                // Original image
                $src = @imagecreatefromwebp($sourceFile);

                break;
            default:
                // Original image
                $src = @imagecreatefromjpeg($sourceFile);

                // Enable interlancing (progressive JPG, smaller size file)
                imageinterlace($dst, true);

                break;
        }

        if ('png' === $extension) {
            imagealphablending($dst, false);
            imagesavealpha($dst, true);
            $transparent = imagecolorallocatealpha($dst, 255, 255, 255, 127);
            imagefilledrectangle($dst, 0, 0, $newWidth, $newHeight, $transparent);
        }

        // Do the resize in memory
        imagecopyresampled($dst, $src, 0, 0, 0, 0, $newWidth, $newHeight, $width, $height);
        imagedestroy($src);

        /*
            Sharpen the image?
            NOTE: requires PHP compiled with the bundled version of GD
            (see http://php.net/manual/en/function.imageconvolution.php)
        */
        if (($this->aiSharpen) && (\function_exists('imageconvolution'))) {
            $intSharpness = $this->findSharp($width, $newWidth);
            $arrMatrix = [
                [
                    -1,
                    -2,
                    -1,
                ],
                [
                    -2,
                    $intSharpness + 12,
                    -2,
                ],
                [
                    -1,
                    -2,
                    -1,
                ],
            ];
            imageconvolution($dst, $arrMatrix, $intSharpness, 0);
        }

        $cacheDir = \dirname($cacheFile);

        // Does the directory exist already?
        if (!is_dir($cacheDir)) {
            if (!mkdir($cacheDir, 0755, true)) {
                // Check again if it really doesn't exist to protect against race conditions
                if (!is_dir($cacheDir)) {
                    // Uh-oh, failed to make that directory
                    imagedestroy($dst);

                    throw new \Exception("Failed to create cache directory: ${cacheDir}");
                }
            }
        }

        if (!is_writable($cacheDir)) {
            throw new \Exception("The cache directory is not writable: ${cacheDir}");
        }

        // Save the new file in the appropriate path, and send a version to the browser
        $cacheFileWebP = $this->webpFilename($cacheFile);

        switch ($extension) {
            case 'png':
                if ($this->aiConvertWebp) {
                    $gotSaved = imagewebp($dst, $cacheFileWebP, $this->aiQuality);
                    $cacheFile = $cacheFileWebP;
                } else {
                    $gotSaved = imagepng($dst, $cacheFile);
                }

                break;
            case 'gif':
                if ($this->aiConvertWebp) {
                    $gotSaved = imagewebp($dst, $cacheFileWebP, $this->aiQuality);
                    $cacheFile = $cacheFileWebP;
                } else {
                    $gotSaved = imagegif($dst, $cacheFile);
                }

                break;
            case 'webp':
                $gotSaved = imagewebp($dst, $cacheFile, $this->aiQuality);

                break;
            default:
                if ($this->aiConvertWebp) {
                    $gotSaved = imagewebp($dst, $cacheFileWebP, $this->aiQuality);
                    $cacheFile = $cacheFileWebP;
                } else {
                    $gotSaved = imagejpeg($dst, $cacheFile, $this->aiQuality);
                }

                break;
        }

        imagedestroy($dst);

        if (!$gotSaved && !file_exists($cacheFile)) {
            throw new \Exception("Failed to create image: ${cacheFile}");
        }

        return $cacheFile;
    }

    /**
     * findSharp - sharpen images function.
     *
     * @param int $intOrig  Params
     * @param int $intFinal Params
     *
     * @return int
     */
    private function findSharp($intOrig, $intFinal)
    {
        $intFinal = $intFinal * (750.0 / $intOrig);
        $intA = 52;
        $intB = -0.27810650887573124;
        $intC = .00047337278106508946;
        $intRes = $intA + $intB * $intFinal + $intC * $intFinal * $intFinal;

        return max(round($intRes), 0);
    }

    private function getResolution($resolutions)
    {
        if (($this->cacheResolution) && ($this->resolution)) {
            return $this->resolution;
        }

        return BrowserFeaturesHelper::create()->getResolution($resolutions);
    }

    private function webpFilename($filename)
    {
        return substr($filename, 0, strrpos($filename, '.')).'.webp';
    }
}
