(function()
{
"use strict";

var _IGNORE_CLASS = "qzhanmkpprf_igore";

var _ignoredTags = [
    "SCRIPT", "IFRAME", "STYLE", "OBJECT", "FOOTER", "NAV" ];

////////////////////////////////////////////////////////////////////////////
// @brief   Singleton Class
////////////////////////////////////////////////////////////////////////////

window.PureView = function()
{
    if( !PureView.prototype._instance )
    {
        PureView.prototype._instance = this;

        // Create and attach the iframe to the body first (which is necessary
        // for modifying its content.)
        this._createAndAppendFrame();
        this._createView();
        this.$frame.contents().find("body").append( this.$view );
    }

    return PureView.prototype._instance;
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._createAndAppendFrame = function()
{
    var style = "\
                display: block; \
                position: fixed; \
                z-index: 2147483647; \
                top: 0; \
                left: 0; \
                width: 100%; \
                height: 100%; \
                touch-action: none; \
                -ms-touch-action: none; \
                overflow:hidden; \
                background-color: gray; \
                ";

    var $frame = jQuery("<iframe" +
                        " frameborder=0" +
                        " style='" + style + "'" +
                        " scrolling='no' seamless='seamless'" +
                    "></iframe>" );

    // Attach the iframe to the DOM tree, but do not display it yet.
    // Note: The iframe has to be part of the DOM tree in order to modify it content.
    $frame.hide();
    jQuery(document.body).append( $frame );

    // Note: The DOCTYPE is necessary for the stylesheet to work.
    var sFrameContent =
        "<!DOCTYPE html>" +
        "<html>" +
            "<head>" +
            "<base target='_parent' />" +
            "<link href='http://fonts.googleapis.com/css?family=Droid+Serif' rel='stylesheet' type='text/css'></link>" +
            "<link href='http://fonts.googleapis.com/css?family=Roboto' rel='stylesheet' type='text/css'></link>" +
            "<link type='text/css' rel='stylesheet' href='/media/plg_pureview/css/pureview.css'></link>" +
            "</head>" +
            "<body>" +
            "</body>" +
        "</html>";

    var eFrameDocument = $frame[0].contentWindow ? $frame[0].contentWindow.document
                                                : $frame[0].contentDocument;

    eFrameDocument.open();
    eFrameDocument.write( sFrameContent );
    eFrameDocument.close();
    
    this.$frame = $frame;
    this.frameDocument = eFrameDocument;
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._createView = function()
{
    var $view = jQuery(
        "<div id='article_view' class='sans_serif'>" +

            "<div id='article_view_dialog'>" +

	            "<div id='article'>" +
	                "<div id='article_title'></div>" +
	                "<div id='article_content'></div>" +
	                "<div id='column_width_ruler'></div>" +
	            "</div>" +
	
	            "<div id='article_view_menu'>" +
	                "<span class='button' id='close_button'></span>" +
	                "<span class='button' id='right_column_button'></span>" +
	                "<span class='button' id='left_column_button'></span>" +
	                "<span class='button' id='full_screen_button'></span>" +
	
	                "<span class='button' id='text_style_button'></span>" +
	                "<div id='text_style_menu' style='display: none;'>" +
	                    "<div class='text_style_menu_section' id='color_scheme_setting'>" +
	                        "<div class='control_label'>Color Scheme</div>" +
	                        "<span class='control_button' id='day_button'>Day</span>" +
	                        "<span class='control_button' id='night_button'>Night</span>" +
	                    "</div>" +
	                    "<div class='text_style_menu_section' id='font_family_setting'>" +
	                        "<div class='control_label'>Font Family</div>" +
	                        "<span class='control_button' id='sans_serif_button'>Sans Serif</span>" +
	                        "<span class='control_button' id='serif_button'>Serif</span>" +
	                    "</div>" +
	                    "<div class='text_style_menu_section' id='font_size_setting'>" +
	                        "<div class='control_label'>Font Size</div>" +
	                        "<span class='control_button' id='decrease_font_size_button'> &#8722; </span>" +  // minus
	                        "<span id='current_font_size'> 15 </span>" +
	                        "<span class='control_button' id='increase_font_size_button'> &#43; </span>" +    // plus
	                    "</div>" +
	                    "<div class='down_arrow'></div>" +
	                "</div>" +
	
	            "</div>" +
	            	
            "</div>" +
            
            "<div id='scrollbar'><div id='scrollbar_handle'></div></div>" +

        "</div>" );

    var $dialog = $view.find( "#article_view_dialog" );
    var $menu   = $view.find( "#article_view_menu" );

    this._initializeViewEventHandling( $view, $dialog );
    this._initializeDialogEventHandling( $dialog );
    this._initializeMenuEventHandling( $menu, $view );

    // Intialize data members.
    this.$view            = $view;
    this.$dialog          = $dialog;
    this.$menu            = $menu;
    this.$article         = $view.find( "#article" );
    this.$articleTitle    = $view.find( "#article_title" );
    this.$articleContent  = $view.find( "#article_content" );
    this.$colWidthRuler   = $view.find( "#column_width_ruler" );
    this.$currentFontSize = $view.find( "#current_font_size" );
    this.$scrollbar       = $view.find( "#scrollbar" );
    this.$scrollbarHandle = $view.find( "#scrollbar_handle" );
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._initializeViewEventHandling = function( $view, $dialog )
{
    var self = this;
/*
    // The view can be closed by clicking on outside the dialog.
    $view.click( function( oEvent )
    {
        if(    $dialog.has( oEvent.target ).length === 0
            && $dialog.is( oEvent.target ) === false )
        {
            self._deactivate();
        }
    });
*/ 
    var isDragging = false, startX = 0;
    $view.find("#scrollbar").mousedown( function(event) 
    {
        startX = event.clientX;
        console.log( "Start: " + startX );
        self.$frame.mousemove(function(event) {
            console.log( "Distance: " + (startX - event.clientX) );
        });
    });
    self.$frame.mouseup( function(event)
    {
        console.log( "End: " + event.clientX )
        self.$frame.unbind("mousemove");
        self._moveColumnHorizontally( 0 );
    });
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._initializeDialogEventHandling = function( $dialog )
{
    var self = this;

    var iPreviousMouseWheelEventTimestamp = 0;
    $dialog.add( self.$frame ).on( "mousewheel", function( oEvent )
    {
        oEvent.preventDefault();
        if( oEvent.timeStamp - iPreviousMouseWheelEventTimestamp > 300 )
        {
            iPreviousMouseWheelEventTimestamp = oEvent.timeStamp;

            ( oEvent.deltaY > 0 ) ? self._moveColumnTowardRight()
                                  : self._moveColumnTowardLeft();
        }
    });

    /* Touch screen event handlers */    
    //if(("ontouchend" in document) || (window.navigator.msPointerEnabled))
    //{
        var iOldLeftScrollPosition = 0;
        $dialog.swipe( { 
            swipeStatus: function( event, phase, direction, distance, duration, fingerCount, fingerData )
            {
                switch( phase )
                {
                    case "start":
                        iOldLeftScrollPosition = self.$dialog.scrollLeft();
                        break;
                    case "move":
                        var iNewLeftScrollPosition = iOldLeftScrollPosition + ( (direction == "left" || direction == "up") ? distance : -distance );
                        self.$dialog.scrollLeft( iNewLeftScrollPosition );
                        break;
                    default:
                    	var speed = distance / duration;
                        if( speed < .2 ) {
                            self._moveColumnHorizontally( 0 );
                        } else {
                            switch( direction )
                            {
                                case "left":
                                    self._moveColumnHorizontally( 1 );
                                    break;
                                case "right":
                                    self._moveColumnHorizontally( -1 );
                                    break;
                                case "up":
                                    self._moveColumnHorizontally( self._getColumnsPerPage() );
                                    break;
                                case "down":
                                    self._moveColumnHorizontally( -self._getColumnsPerPage() );
                            }
                        }
                }
            },
            pinchStatus: function( event, direction, distance, duration, fingerCount, zoom, fingerData )
            {
                console.log( zoom );
            },
            allowPageScroll:"none",
            fallbackToMouseEvents:false,
            threshold:200,
            maxTimeThreshold:5000
        });
    //}

    // Resize the dialog every time the window is resized.
    jQuery(window).resize( function()
    {
        self._maximizeAndCenterDialog();
    });
    
    // Add keyboard shortcuts.      
    jQuery(document).keydown( function (e) 
    {
        if( self._isActivated() )
            self._keyHandler( e.keyCode, self );
    });

    this.$frame.keydown ( function (e) 
    {
        if( self._isActivated() )
            self._keyHandler( e.keyCode, self );
    });
};



////////////////////////////////////////////////////////////////////////////

PureView.prototype._initializeMenuEventHandling = function( $menu, $view )
{
    var self = this;

    // Find all the control elements.

    var $closeButton            = $menu.find( "#close_button"        );
    var $rightColButton         = $menu.find( "#right_column_button" );
    var $leftColButton          = $menu.find( "#left_column_button"  );
    var $fullScrButton          = $menu.find( "#full_screen_button"  );
    var $textStyleButton        = $menu.find( "#text_style_button"   );

    var $textStyleMenu          = $menu.find( "#text_style_menu"           );
    var $increaseFontSizeButton = $menu.find( "#increase_font_size_button" );
    var $decreaseFontSizeButton = $menu.find( "#decrease_font_size_button" );
    var $sansSerifButton        = $menu.find( "#sans_serif_button"         );
    var $serifButton            = $menu.find( "#serif_button"              );
    var $dayButton              = $menu.find( "#day_button"                );
    var $nightButton            = $menu.find( "#night_button"              );

    // Add menu button handlers.

               $closeButton.click( function() { self._deactivate();                     } );
            $rightColButton.click( function() { self._moveColumnTowardLeft();           } );
             $leftColButton.click( function() { self._moveColumnTowardRight();          } );
             $fullScrButton.click( function() { self._toggleFullScreen($fullScrButton); } );

    $increaseFontSizeButton.click( function() { self._increaseFontSize(); } );
    $decreaseFontSizeButton.click( function() { self._decreaseFontSize(); } );

           $sansSerifButton.click( function() { self._setSansSerifFont(); } );
               $serifButton.click( function() { self._setSerifFont();     } );

                 $dayButton.click( function() { self._setDayScheme();   } );
               $nightButton.click( function() { self._setNightScheme(); } );

    // Add menu activation/deactivation handlers.

    $textStyleButton.click( function()
    {
        $textStyleMenu.fadeToggle();
        return false;
    } );

    $view.click( function( oEvent )
    {
        if(    $textStyleMenu.has( oEvent.target ).length === 0
            && $textStyleMenu.is( oEvent.target ) === false )
        {
            $textStyleMenu.fadeOut();
        }
    } );

};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._keyHandler = function( iKeyCode, self )
{

    var  LEFT_ARROW_KEY = 37,
        RIGHT_ARROW_KEY = 39,
           UP_ARROW_KEY = 38,
         DOWN_ARROW_KEY = 40,
               PGUP_KEY = 33,
               PGDN_KEY = 34,
               HOME_KEY = 36,
                END_KEY = 35,
                ESC_KEY = 27;
   
    console.log( iKeyCode );
    switch( iKeyCode )
    {
        case LEFT_ARROW_KEY:
            self._moveColumnHorizontally( -1 );
            return false;
        case UP_ARROW_KEY:
        case PGUP_KEY:
            self._moveColumnHorizontally( -self._getColumnsPerPage() );
            return false;
        case RIGHT_ARROW_KEY:
            self._moveColumnHorizontally( 1 );
            return false;
        case DOWN_ARROW_KEY:
        case PGDN_KEY:
            self._moveColumnHorizontally( self._getColumnsPerPage() );
            return false;
        case ESC_KEY:
            self._deactivate();
            return false;
	    case HOME_KEY:
	        self._moveHorizontally( 0 );
	        return false;
            case END_KEY:
	        self._moveHorizontally( self.$dialog[0].scrollWidth );
	        return false;
        default:
            // Nothing
    }
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._toggleFullScreen = function( eFullScreenButton )
{
    var elem = this.$frame[0];
    if ((document.fullScreenElement && document.fullScreenElement !== null) || 
        (!document.msFullscreenElement && !document.mozFullScreenElement && !document.webkitFullscreenElement))
    {
        if (elem.requestFullscreen) {
            elem.requestFullscreen();
        } else if (elem.msRequestFullscreen) {
            elem.msRequestFullscreen();
        } else if (elem.mozRequestFullScreen) {
           elem.mozRequestFullScreen();
        } else if (elem.webkitRequestFullscreen) {
           elem.webkitRequestFullscreen();
        }
        eFullScreenButton.addClass( "full" );
    } else {
        if (document.exitFullscreen) {
            document.exitFullscreen();
        } else if (document.webkitExitFullscreen) {
            document.webkitExitFullscreen();
        } else if (document.mozCancelFullScreen) {
            document.mozCancelFullScreen();
        } else if (document.msExitFullscreen) {
            document.msExitFullscreen();
        }
        eFullScreenButton.removeClass( "full" );
    }
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._changeFontSize = function( iDiff )
{
    var $html            = this.$view.closest("html");
    var iRootFontSize    = parseInt( $html.css( "font-size" ), 10 );
    var iNewRootFontSize = Math.max( 8, iRootFontSize + iDiff );  // lower bound

    $html.css( "font-size", iNewRootFontSize + "px" );

    localStorage[ "PurifySettingRootFontSize" ] = iNewRootFontSize;
    
    this.$currentFontSize.text( iNewRootFontSize );

    this._maximizeAndCenterDialog();
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._increaseFontSize = function()
{
    this._changeFontSize( 1 );
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._decreaseFontSize = function()
{
    this._changeFontSize( -1 );
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._setFontFamily = function( sFontFamilyClass )
{
    this.$view.removeClass( "serif sans_serif" );
    this.$view.addClass( sFontFamilyClass );

    localStorage[ "PurifySettingRootFontFamily" ] = sFontFamilyClass;

    this._maximizeAndCenterDialog();
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._setSansSerifFont = function()
{
    this._setFontFamily( "sans_serif" );
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._setSerifFont = function()
{
    this._setFontFamily( "serif" );
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._setColorScheme = function( sColorScheme )
{
    this.$view.removeClass( "day night" );
    this.$view.addClass( sColorScheme );

    localStorage[ "PurifySettingRootColorScheme" ] = sColorScheme;
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._setDayScheme = function()
{
    this._setColorScheme( "day" );
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._setNightScheme = function()
{
    this._setColorScheme( "night" );
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype.activate = function()
{
    var self            = this;
    var oArticle        = (new ArticleParser()).getArticle();
    var $articleTitle   = oArticle.$title;
    var $articleContent = oArticle.$content;

    if( $articleContent.length > 0 )
    {   
        self._applySavedSetting();

        self.$dialog.hide();
        self.$article.hide();

        self.$frame.fadeIn( "fast", function()
        {
            jQuery("html").css( "overflow", "hidden" );  // Hide the scrollbar
	    jQuery("body").css( "-ms-overflow-style", "scrollbar" );  // Hide the scrollbar

            self._maximizeAndCenterDialog();
            self.$dialog.fadeIn( "fast", function()
            {
                // Note: jQuery.append() might generate syntax error.

                if( $articleTitle[0] )   
                {
                    self.$articleTitle.html( "" );
                    self.$articleTitle[0].appendChild( $articleTitle[0] );
                }

                if( $articleContent[0] )
                {
                    self.$articleContent.html( "" );
                    self.$articleContent[0].appendChild( $articleContent[0] );
                }
              
                self.$article.fadeIn( function() {
                    var iMaxOffset = self.$dialog[0].scrollWidth - self.$dialog.width();
                    var iTotalColumnWidth = self._getColumnWidth() + self._getColumnGap();
                    var iColumnCount = Math.round(iMaxOffset / iTotalColumnWidth) + 1;                   
                    self.$scrollbarHandle.css( "width", 100 / iColumnCount + "%" );
                } );
            });
        });
    }
    else
    {
        // Content is always available even if this is not an article.
    }
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._applySavedSetting = function()
{
    var self = this;

    var value = localStorage[ "PurifySettingRootFontSize" ];
    if( value )
    {
        var iCurrentFontSize = value;
        var $html = self.$view.closest("html");

        $html.css( "font-size", iCurrentFontSize + "px" );
        self.$currentFontSize.text( iCurrentFontSize );
    }

    var value = localStorage[ "PurifySettingRootFontFamily" ];
    if( value )
    {
        self.$view.removeClass( "serif sans_serif" );
        self.$view.addClass( value );
    }
    
    var value = localStorage[ "PurifySettingRootColorScheme" ];
    if( value )
    {
        self.$view.removeClass( "day night" );
        self.$view.addClass( value );
    }

};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._maximizeAndCenterDialog = function()
{
    var $container = this.$dialog;
    var $window = jQuery(window);

    var iWindowWidth  = window.innerWidth  || $window.width();
    var iWindowHeight = window.innerHeight || $window.height();  // $window.height() could return a much bigger value.

    var iHorizontalPadding = $container.outerWidth()  - $container.width();
    var iVerticalPadding   = $container.outerHeight() - $container.height();

    var MARGIN = Math.min(20, Math.floor(iWindowWidth * 0.01), Math.floor(iWindowHeight * 0.01));   // px; top + bottom; left + right

    $container.width(  iWindowWidth  - (MARGIN * 2) - iHorizontalPadding );
    $container.height( iWindowHeight - (MARGIN * 2) - iVerticalPadding   );

    $container.css( { "left" : MARGIN + "px",
                      "top"  : MARGIN + "px" } );
    
    this.$scrollbar.width( iWindowWidth  - (MARGIN * 2) );                  
    this.$scrollbar.css( { "left"   : MARGIN + "px",
                           "bottom" : MARGIN + "px" } );
                                 
    this._moveColumnHorizontally( 0 );
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._deactivate = function()
{
    jQuery("html").css( "overflow", "" );
    jQuery("body").css( "-ms-overflow-style", "" );
    
    this.$frame.fadeOut( "", function()
    {
        if (document.exitFullscreen) {
            document.exitFullscreen();
        } else if (document.webkitExitFullscreen) {
            document.webkitExitFullscreen();
        } else if (document.mozCancelFullScreen) {
            document.mozCancelFullScreen();
        } else if (document.msExitFullscreen) {
            document.msExitFullscreen();
        }
    });
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._isActivated = function()
{
    return this.$frame.css( "display" ) !== "none";
}


////////////////////////////////////////////////////////////////////////////

PureView.prototype._moveHorizontally = function( iNewLeftScrollPosition )
{
    var iLeftScrollPosition = this.$dialog.scrollLeft()
    var iMaxOffset = this.$dialog[0].scrollWidth - this.$dialog.width();
    
    iNewLeftScrollPosition = Math.max( 0, Math.min( iMaxOffset, iNewLeftScrollPosition ) );
    
    if( iNewLeftScrollPosition != iLeftScrollPosition )
    {
      
        this.$dialog
          .animate( { "scrollLeft" : iNewLeftScrollPosition },
                      "fast",
                      "linear" );
                      
        var iTotalColumnWidth = this._getColumnWidth() + this._getColumnGap();
        var iColumnCount = Math.round(iMaxOffset / iTotalColumnWidth) + 1;
        
        this.$scrollbarHandle
          .animate({ "width"      : 100 / iColumnCount + "%",
                     "marginLeft" : Math.round( iNewLeftScrollPosition / iTotalColumnWidth) / iColumnCount * 100 + "%" },
                     "fast",
                     "linear" );
    }
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._moveColumnHorizontally = function( iColumns )
{
    var iLeftScrollPosition = this.$dialog.scrollLeft();
    var iTotalColumnWidth   = this._getColumnWidth() + this._getColumnGap();
    
    if( iTotalColumnWidth > 0 )
    {
        var iNewLeftScrollPosition = (Math.round(iLeftScrollPosition / iTotalColumnWidth) + iColumns) * iTotalColumnWidth;
        this._moveHorizontally( iNewLeftScrollPosition );
    }
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._moveColumnTowardLeft = function()
{
    this._moveColumnHorizontally( 1 );
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._moveColumnTowardRight = function()
{
    this._moveColumnHorizontally( -1 );
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._getColumnsPerPage = function()
{
    return Math.round( this.$dialog.width() / this.$colWidthRuler.width() );  // integer
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._getColumnWidth = function()
{
    return this.$colWidthRuler.width();  // integer
};


////////////////////////////////////////////////////////////////////////////

PureView.prototype._getColumnGap = function()
{
    return parseInt( this.$article.css("margin-right"), 10);
};




////////////////////////////////////////////////////////////////////////////
// @brief   Singleton Class
////////////////////////////////////////////////////////////////////////////

window.ArticleParser = function() {};   // Empty Constructor.


////////////////////////////////////////////////////////////////////////////

ArticleParser.prototype.getArticle = function()
{
    var $contentContainer = this._findContentContainer();

    this._revertModificationToExistingArticle();

    return { "$title"   : this._getTitle( $contentContainer ),
            "$content" : this._getContent( $contentContainer ) };
};


////////////////////////////////////////////////////////////////////////////

ArticleParser.prototype._revertModificationToExistingArticle = function()
{
    jQuery("." + _IGNORE_CLASS).removeClass( _IGNORE_CLASS );
};


////////////////////////////////////////////////////////////////////////////

ArticleParser.prototype._findContentContainer = function()
{
    var $contentContainer = jQuery( ".article-content" );

    return $contentContainer;
};



////////////////////////////////////////////////////////////////////////////

ArticleParser.prototype._shouldIgnoreElement = function( eElement )
{

    var shouldIgnore = false;

    if( eElement )
    {
        var $element = jQuery( eElement );

        try
        {
            shouldIgnore = (
                $element.css( "float" ) === "left"
            || $element.css( "float" ) === "right"
            || $element.css( "position" ) === "absolute"
            || $element.hasClass( _IGNORE_CLASS )
            || _ignoredTags.some( function( sTagName ) { return eElement.tagName === sTagName; } ) );
        }
        catch( exception )
        {
            // jQuery will throw exception probably because it is not fully
            // tested in the Chrome extension environment.
        }
    }

    return shouldIgnore;
};



////////////////////////////////////////////////////////////////////////////

ArticleParser.prototype._getTitle = function( $contentContainer )
{
    var $title = jQuery();

    if( $contentContainer.length > 0 )
    {
        var $originalPrimaryTitle = this._getPrimaryTitle( $contentContainer );

        var $primaryTitle = $originalPrimaryTitle.clone();
        $primaryTitle.find( ".pureViewLink" ).remove();
        
        this._sanitizeElementAndDesendants( $primaryTitle );

        $title = jQuery("<div></div>");
        $title.append( $primaryTitle );
    }

    return $title;
};


////////////////////////////////////////////////////////////////////////////

ArticleParser.prototype._getHeadingElementClosestToContainer = function( eContentContainer )
{
    // Find the heading elment that is closest to the content container.

    var $title = jQuery()

    var listOfHeadingElementTags = [ "h1", "h2", "h3" ];
    var listOfHeadingElements = [];

    for( var i = 0;
            i < listOfHeadingElementTags.length
        && listOfHeadingElements.length === 0;
        i++ )
    {
        listOfHeadingElements = jQuery.makeArray( jQuery( listOfHeadingElementTags[i] ) );
    }

    listOfHeadingElements.sort( function( eElement1, eElement2 )
    {
        var iDistance1 = Util.getDistanceBetweenElements( eElement1, eContentContainer );
        var iDistance2 = Util.getDistanceBetweenElements( eElement2, eContentContainer );

        return iDistance1 - iDistance2;
    });

    var eHeadingElementClosestToContent = listOfHeadingElements[0];
    if( eHeadingElementClosestToContent )
    {
        $title = jQuery( eHeadingElementClosestToContent );

        // Prevent dupolicate titles from appearing in the content.
        jQuery(eHeadingElementClosestToContent).addClass( _IGNORE_CLASS );
    }

    return $title;
}



////////////////////////////////////////////////////////////////////////////

ArticleParser.prototype._getPrimaryTitle = function( $contentContainer )
{
    var $title = jQuery();
    var eContentContainer = $contentContainer[0];

    if( eContentContainer )
    {

        $title = jQuery( ".article-title" );

        if( $title.length === 0 )
        {
            $title = this._getHeadingElementClosestToContainer( eContentContainer )
        }

        // For styling purposes, make sure the title is a <h1> element.
        if( $title.length > 0 && $title[0].tagName !== "H1" )
        {
            var $h1 = jQuery("<h1></h1>");
            $h1.html( $title.html() );
            $title = $h1;
        }
    }

    return $title;
};



////////////////////////////////////////////////////////////////////////////

ArticleParser.prototype._getContent = function( $contentContainer )
{
    var $result = jQuery();

    if( $contentContainer.length > 0 )
    {
        // Note: clone() does not clone the computed style. Need to traverse
        //       the original container to analyze their styles.

        var $contentContainerClone = $contentContainer.clone();
        
        this._restoreSeparatedParagraphs( $contentContainer,
                                        $contentContainerClone );

        var listOfExistingElements = [ $contentContainer[0] ];
        var listOfClonedElements = [ $contentContainerClone[0] ];

        while( listOfClonedElements.length > 0 )
        {
            var eExistingElement = listOfExistingElements.shift();
            var eClonedElement = listOfClonedElements.shift();

            if(    eExistingElement !== $contentContainer[0]
                && this._shouldIgnoreElement( eExistingElement ) )
            {
            jQuery(eClonedElement).remove();
            }
            else
            {
            var isRemoved = this._sanitizeElement( eClonedElement );
            if( !isRemoved )
            {
                if( eClonedElement.childNodes.length ===
                    eExistingElement.childNodes.length )
                {
                    for( var i = 0; i < eClonedElement.childNodes.length; i++ )
                    {
                        listOfExistingElements.push( eExistingElement.childNodes[i] );
                        listOfClonedElements.push( eClonedElement.childNodes[i] );
                    }
                }
            }
            }
        }

        // Use appendChild() instead of jQuery.append() to add the content instead
        // to prevent runtime errors.

        $result = jQuery("<div></div>");
        $result[0].appendChild( $contentContainerClone[0] );
    }

    return $result;
};



////////////////////////////////////////////////////////////////////////////

ArticleParser.prototype._restoreSeparatedParagraphs = function( $contentContainer,
                                                                $contentContainerClone )
{
    // Handle the special cases where the article content is split into
    // two separate containers.

    var isSplitIntoTwoContainers = false;
    var listOfContainers = [ $contentContainer, $contentContainer.parent() ];

    for( var i = 0; i < listOfContainers.length && !isSplitIntoTwoContainers; i++ )
    {
        var $container = listOfContainers[i];
        var sClassAttr = $container.attr( "class" );
        var listOfClassNames = sClassAttr ? sClassAttr.split( " " ) : [];

        for( var j = 0; j < listOfClassNames.length && !isSplitIntoTwoContainers; j++ )
        {
            var sClassName = listOfClassNames[j];

            if(    Util.hasSubstringCaseInsensitive( sClassName, "article" )
                && Util.hasSubstringCaseInsensitive( sClassName, "body" ) )
            {
            var $splitContentContainer = $container.siblings( "." + sClassName );
            var $separatedParagraphs = $splitContentContainer.find( "p" );

            if(    $splitContentContainer.length === 1
                && $separatedParagraphs.length > 0 )
            {
                isSplitIntoTwoContainers = true;

                $separatedParagraphs.each( function( iIndex, eParagraph )
                {
                    var $clonedParagraph = jQuery(eParagraph).clone();
                    $contentContainerClone.prepend( $clonedParagraph );

                    // Need to make sure the original content container has
                    // the same number of elements as the clone.
                    var $clonedParagraph2 = jQuery(eParagraph).clone();
                    $clonedParagraph2.css( "display", "none" );
                    $contentContainer.prepend( $clonedParagraph2 );
                });
            }
            }
        }
    }
};



////////////////////////////////////////////////////////////////////////////

ArticleParser.prototype._sanitizeElement = function( eElement )
{
    if( eElement )
    {
        eElement.id        = "";
        eElement.className = "";

        if( eElement.removeAttribute )
        {
            eElement.removeAttribute( "style" );
            eElement.removeAttribute( "width" );
            eElement.removeAttribute( "height" );
        }
    }
};



////////////////////////////////////////////////////////////////////////////

ArticleParser.prototype._sanitizeElementAndDesendants = function( element )
{
    var eElement = ( element instanceof jQuery ) ? element[0] : element;

    if( eElement )
    {
        var listOfElements = [ eElement ];

        while( listOfElements.length > 0 )
        {
            var e = listOfElements.shift();

            this._sanitizeElement( e );

            for( var i = 0; i < e.childNodes.length; i++ )
            {
                listOfElements.push( e.childNodes[i] );
            }
        }
    }
};


window.Util = {};

////////////////////////////////////////////////////////////////////////////

Util.getDistanceBetweenElements = function( eElement1, eElement2 )
{
    var getPosition = function( eElement )
    {
        var oResult = { "top"  : 0,
                        "left" : 0 };

        while( eElement )
        {
            oResult.top  += eElement.offsetTop;
            oResult.left += eElement.offsetLeft;

            eElement = eElement.offsetParent;
        }

        return oResult;
    };

    var oPosition1 = getPosition( eElement1 );
    var oPosition2 = getPosition( eElement2 );

    var y = oPosition1.top  - oPosition2.top;
    var x = oPosition1.left - oPosition2.left;

    return Math.sqrt( y * y + x * x );
};



////////////////////////////////////////////////////////////////////////////

Util.hasSubstringCaseInsensitive = function( string, substring )
{
    var stringCopy = string.toLowerCase();
    var substringCopy = substring.toLowerCase();

    return ( stringCopy.indexOf( substringCopy ) !== -1 );
}


})();