<?php
/* This file has been prefixed by <PHP-Prefixer> for "XT Adaptive Images" */

/*
 * @package     XT Adaptive Images
 *
 * @author      Extly, CB. <team@extly.com>
 * @copyright   Copyright (c)2012-2022 Extly, CB. All rights reserved.
 * @license     https://www.gnu.org/licenses/gpl-3.0.html GNU/GPL
 *
 * @see         https://www.extly.com
 */

defined('_JEXEC') || exit;

use XTP_BUILD\Extly\AdaptiveImages\Common\ImageProcessor;
use XTP_BUILD\Extly\AdaptiveImages\Common\LazyLoadHelper;
use XTP_BUILD\Extly\AdaptiveImages\Common\SrcsetHelper;
use XTP_BUILD\Extly\Infrastructure\Service\Cms\Joomla\ScriptHelper;
use Joomla\CMS\Factory as CMSFactory;
use Joomla\CMS\HTML\HTMLHelper as CMSHTMLHelper;
use Joomla\CMS\Uri\Uri as CMSUri;

/**
 * AdaptiveImagesForJoomlaHelper class - Plugin that replaces media urls with Adaptive Images urls.
 *
 * @since       1.0
 */
class AdaptiveImagesForJoomlaHelper
{
    const LAZYLOAD_LIBRARY_J_QUERY_LAZY_LOAD = 0;

    const LAZYLOAD_LIBRARY_VANILLA_LAZY_LOAD = 1;

    const LAZYLOAD_LIBRARY_LAZYSIZES = 2;

    private $params;

    private $pass = false;

    private $currentSet;

    private $aiCachePath;

    private $imageIgnoreFiles;

    private $hasCdn = false;

    private $isSsl = false;

    private $srcsetDictionary = [];

    /**
     * AdaptiveImagesForJoomlaHelper.
     *
     * @param object &$params Param
     */
    public function __construct(&$params)
    {
        $this->params = $params;
        $this->hasCdn = preg_replace(['#^.*\://#', '#/$#'], '', $this->getCdnParam());
        $this->params->set('tag_attribs', $this->getSearchTagAttributes());

        $this->isSsl = (
            (isset($_SERVER['HTTPS']) && 'ON' === strtoupper($_SERVER['HTTPS']))
            || (isset($_SERVER['SERVER_PORT']) && 443 === strtoupper($_SERVER['SERVER_PORT']))
            || (0 === strpos(CMSUri::root(), 'https://'))
        );

        $this->initSetList();
        $sets = $this->params->get('sets');

        if (empty($sets)) {
            return;
        }

        $this->pass = true;
        $this->aiCachePath = $params->get('ai_cache_path', 'media/xt-adaptive-images');

        $this->imageIgnoreFiles = explode(',', str_replace(['\n', ' '], [',', ''], $this->params->get('image_ignorefiles')));

        if (!@include_once(JPATH_ROOT.'/libraries/adaptiveimagesforjoomla/vendor/autoload.php')) {
            return;
        }
    }

    public function addScripts()
    {
        if ($this->params->get('retina_displays')) {
            ScriptHelper::addScriptDeclaration("document.cookie='resolution='+Math.max(screen.width,screen.height)+(\"devicePixelRatio\" in window ? \",\"+devicePixelRatio : \",1\")+'; path=/';");
        } else {
            ScriptHelper::addScriptDeclaration("document.cookie='resolution='+Math.max(screen.width,screen.height)+'; path=/';");
        }

        if ($this->params->get('enable_lazyload')) {
            $detectionClass = $this->params->get('detection_class', 'xt-lazy-img');
            $detectionClass = str_replace(' ', '', $detectionClass);
            $classes = explode(',', $detectionClass);

            if (empty($classes)) {
                $this->params->set('enable_lazyload', false);

                return;
            }

            $this->params->set('lazyload_classes', $classes);
            $this->defineLazyloadLibrary(LazyLoadHelper::LAZYLOAD_SELECTOR);
        }
    }

    /**
     * apply.
     */
    public function apply()
    {
        if (!$this->pass) {
            return;
        }

        $html = CMSFactory::getApplication()->getBody();

        $this->replace($html);

        if (($this->params->get('generate_srcset')) && (!empty($this->srcsetDictionary))) {
            $srcsetHelper = new SrcsetHelper();
            $srcsetHelper->setSrcsetDictionary($this->srcsetDictionary);
            $srcsetHelper->setSizes($this->params->get('generate_srcset_sizes', '100vw'));
            $srcsetHelper->addSrcsetSizes($html, $this);
        }

        if ($this->params->get('enable_lazyload')) {
            $lazyLoadHelper = new LazyLoadHelper($this);

            // LAZYLOAD_LIBRARY_VANILLA_LAZY_LOAD or LAZYLOAD_LIBRARY_LAZYSIZES
            if ((int) $this->params->get('lazyload_library')) {
                // Same attributes
                $lazyLoadHelper->setLibMode(LazyLoadHelper::VANILLA_LAZY_LOAD);
            }

            $lazyLoadHelper->replaceLazyLoad(
                $html,
                $this->params->get('lazyload_classes'),
                $this->params->get('skip_images', 0)
            );
        }

        $this->cleanLeftoverJunk($html);

        CMSFactory::getApplication()->setBody($html);
    }

    /**
     * fileIsIgnored.
     *
     * @param string $file Param
     *
     * @return bool
     */
    public function fileIsIgnored($file)
    {
        foreach ($this->currentSet->ignorefiles as $ignore) {
            if (
                $ignore &&
                (false !== strpos($file, $ignore) || false !== strpos(htmlentities($file), $ignore))
            ) {
                return true;
            }
        }

        return false;
    }

    /**
     * filetypeIsIgnored.
     *
     * @param string $file Param
     *
     * @return bool
     */
    public function filetypeIsIgnored($file)
    {
        return !in_array(pathinfo($file, \PATHINFO_EXTENSION), $this->currentSet->filetypes, true);
    }

    /**
     * imageFileIsIgnored.
     *
     * @param string $file Param
     *
     * @return bool
     */
    public function imageFileIsIgnored($file)
    {
        foreach ($this->imageIgnoreFiles as $ignore) {
            if (
                $ignore && (false !== strpos($file, $ignore) || false !== strpos(htmlentities($file), $ignore))
            ) {
                return true;
            }
        }

        return false;
    }

    /**
     * protectString.
     *
     * @param string $string Param
     */
    public function protectString($string)
    {
        if ($this->isEditPage()) {
            $string = preg_replace('#(<'.'form [^>]*(id|name)="(adminForm|postform)".*?</form>)#si', '{nocdn}\1{/nocdn}', $string);
        }

        if (false === strpos($string, '{nocdn}') || false === strpos($string, '{/nocdn}')) {
            $string = str_replace(['{nocdn}', '{/nocdn}'], '', $string);

            return [$string];
        }

        $string = str_replace(['{nocdn}', '{/nocdn}'], '[[CDN_SPLIT]]', $string);

        return explode('[[CDN_SPLIT]]', $string);
    }

    /**
     * isAdmin - check if page is an admin page.
     *
     * @param bool $blockLogin Param
     *
     * @return bool
     */
    protected function isAdmin($blockLogin = false)
    {
        return
                CMSFactory::getApplication()->isClient('administrator')
                && (!$blockLogin || 'com_login' !== CMSFactory::getApplication()->input->get('option'))
                && 'preview' !== CMSFactory::getApplication()->input->get('task')
        ;
    }

    protected function getCdnParam($setid = null)
    {
        $cdn = $this->params->get('cdn'.$setid);

        if ((!$setid) && (empty($cdn))) {
            return CMSUri::root();
        }

        return $cdn;
    }

    private function defineLazyloadLibrary($selectors)
    {
        $lazyloadLibrary = (int) $this->params->get('lazyload_library', self::LAZYLOAD_LIBRARY_LAZYSIZES);

        if (self::LAZYLOAD_LIBRARY_VANILLA_LAZY_LOAD === $lazyloadLibrary) {
            // vanilla-lazyload
            // https://github.com/verlok/vanilla-lazyload/
            // https://www.andreaverlicchi.eu/vanilla-lazyload-vs-lazysizes/#:~:text=What%20are%20vanilla%2Dlazyload%20and,bit%20earlier%20than%20they%20do.
            ScriptHelper::addStyleDeclaration('img:not([src]){visibility:hidden;}@-moz-document url-prefix(){img:-moz-loading{visibility:hidden;}}');
            ScriptHelper::addDeferredExtensionScript('plg_system_adaptiveimagesforjoomla/lazyload.min.js');
            ScriptHelper::addScriptDeclaration('window.lazyLoadOptions={elements_selector:"'.$selectors.'"};');

            return;
        }

        if (self::LAZYLOAD_LIBRARY_J_QUERY_LAZY_LOAD === $lazyloadLibrary) {
            CMSHTMLHelper::_('jquery.framework');
            // https://www.appelsiini.net/projects/lazyload/
            ScriptHelper::addDeferredExtensionScript('plg_system_adaptiveimagesforjoomla/jquery.lazyload.min.js');
            ScriptHelper::addScriptDeclaration('jQuery(document).ready(function(){jQuery("'.$selectors.'").lazyload()});');

            return;
        }

        // Fade image in after load
        // https://github.com/aFarkas/lazysizes
        // https://web.dev/use-lazysizes-to-lazyload-images/
        ScriptHelper::addStyleDeclaration('.lazyload,.lazyloading{opacity: 0;}.lazyloaded{opacity: 1;transition: opacity 150ms;}');
        ScriptHelper::addDeferredExtensionScript('plg_system_adaptiveimagesforjoomla/lazysizes.min.js');
        ScriptHelper::addScriptDeclaration('document.addEventListener("DOMContentLoaded",function(){window.lazySizes&&document.querySelectorAll("'.$selectors.'").forEach(function(a){a.classList.add("lazyload")})});');
    }

    /**
     * getSearchTagAttributes.
     *
     * @return string
     */
    private function getSearchTagAttributes()
    {
        $attributes = [
            'href=',
            'src=',
            'data-original=',
            'longdesc=',
            'poster=',
            '@import',
            'name="movie" value=',
            'property="og:image" content=',
            'rel="{\'link\':',
        ];

        return str_replace(['"', '=', ' '], ['["\']?', '\s*=\s*', '\s+'], implode('|', $attributes));
    }

    /**
     * initSetList.
     */
    private function initSetList()
    {
        $sets = [];

        // Adaptiveimagesforjoomla-free
        $nrOfSets = 1;

        for ($i = 1; $i <= $nrOfSets; ++$i) {
            $sets[] = $this->initSetParams($i);
        }

        $this->params->set('sets', $sets);

        $this->removeEmptySets();
    }

    /**
     * initSetParams.
     *
     * @param int $setid Param
     */
    private function initSetParams($setid = 1)
    {
        $index = $setid;
        $setid = ($setid <= 1) ? '' : '_'.(int) $setid;

        if (($index > 1) && (!preg_replace(['#^.*\://#', '#/$#'], '', $this->getCdnParam($setid)))) {
            return false;
        }

        $enableHttps = $this->params->get('enable_https'.$setid, true);

        // If is SSL and not enabled SSL at CDN, avoid any processing
        if (($this->isSsl) && (!$enableHttps)) {
            return false;
        }

        /*
        $filetypes = str_replace('-', ',', implode(',', $this->params->get('{'filetypes' . $setid}));
        $filetypes = explode(',', $filetypes);
        */

        $filetypes = [];

        if ($this->params->get('gif'.$setid, true)) {
            $filetypes[] = 'gif';
        }

        if ($this->params->get('jpg_jpeg'.$setid, true)) {
            $filetypes[] = 'jpg';
            $filetypes[] = 'jpeg';
        }

        if ($this->params->get('png'.$setid, true)) {
            $filetypes[] = 'png';
        }

        if ($this->params->get('svg'.$setid, true)) {
            $filetypes[] = 'svg';
        }

        if ($this->params->get('webp'.$setid, true)) {
            $filetypes[] = 'webp';
        }

        $extratypes = preg_replace('#\s#', '', $this->params->get('extratypes'.$setid));

        if ($extratypes) {
            $filetypes = array_merge($filetypes, explode(',', $extratypes));
        }

        $filetypes = array_unique(array_diff($filetypes, ['', 'x']));

        if (empty($filetypes)) {
            return false;
        }

        $currentSet = new stdClass();
        $currentSet->filetypes = $filetypes;

        $currentSet->enable_in_scripts = $this->params->get('enable_in_scripts'.$setid, false);

        // Adaptiveimagesforjoomla-free
        $currentSet->protocol = "http://";

        $currentSet->cdn = preg_replace('#/$#', '', $this->getCdnParam($setid));
        $currentSet->enable_https = $enableHttps;
        $currentSet->ignorefiles = explode(',', str_replace(['\n', ' '], [',', ''], $this->params->get('ignorefiles'.$setid)));

        $root = preg_replace(['#^/#', '#/$#'], '', $this->params->get('root'.$setid)).'/';
        $currentSet->root = preg_replace('#^/#', '', $root);

        $currentSet->searches = [];
        $currentSet->jsSearches = [];

        $this->currentSet = $currentSet;

        $this->setFiletypeSearches();
        $this->setCdnPaths();

        return $this->currentSet;
    }

    /**
     * removeEmptySets.
     */
    private function removeEmptySets()
    {
        $sets = $this->params->get('sets');

        foreach ($sets as $i => $set) {
            if ((!empty($set)) && (!empty($set->searches))) {
                continue;
            }

            unset($sets[$i]);
        }

        $this->params->set('sets', $sets);
    }

    /**
     * replace.
     *
     * @param string &$string Param
     */
    private function replace(&$string)
    {
        if (is_array($string)) {
            $this->replaceInList($string);

            return;
        }

        $stringArray = $this->protectString($string);

        foreach ($stringArray as $i => &$substring) {
            if ($i % 2) {
                continue;
            }

            $stringArray[$i] = $this->replaceBySetList($substring);
        }

        $string = implode('', $stringArray);
    }

    /**
     * replaceInList.
     *
     * @param array &$array Param
     */
    private function replaceInList(&$array)
    {
        foreach ($array as &$val) {
            $this->replace($val);
        }
    }

    /**
     * replaceBySetList.
     *
     * @param string $string Param
     *
     * @return string
     */
    private function replaceBySetList($string)
    {
        $sets = $this->params->get('sets');

        foreach ($sets as $set) {
            $this->currentSet = $set;
            $this->replaceBySet($string);
        }

        return $string;
    }

    /**
     * replaceBySet.
     *
     * @param string &$string Param
     */
    private function replaceBySet(&$string)
    {
        $this->replaceBySearchList($string, $this->currentSet->searches);

        if ((!empty($this->currentSet->enable_in_scripts))
            && (false !== strpos($string, '<script'))) {
            $this->replaceInJavascript($string);
        }
    }

    /**
     * replaceInJavascript.
     *
     * @param string &$string Param
     */
    private function replaceInJavascript(&$string)
    {
        $regex = '#<script(?:\s+(language|type)\s*=[^>]*)?>.*?</script>#si';

        if (preg_match_all($regex, $string, $stringparts, \PREG_SET_ORDER) < 1) {
            return;
        }

        foreach ($stringparts as $stringpart) {
            $this->replaceInJavascriptStringPart($string, $stringpart);
        }
    }

    /**
     * replaceInJavascriptStringPart.
     *
     * @param string &$string    Param
     * @param string $stringpart Param
     */
    private function replaceInJavascriptStringPart(&$string, $stringpart)
    {
        $newstr = $stringpart['0'];

        if (!$this->replaceBySearchList($newstr, $this->currentSet->jsSearches)) {
            return;
        }

        $string = str_replace($stringpart['0'], $newstr, $string);
    }

    /**
     * replaceBySearchList.
     *
     * @param string &$string   Param
     * @param array  &$searches Param
     */
    private function replaceBySearchList(&$string, &$searches)
    {
        $changed = 0;

        foreach ($searches as $search) {
            $changed = $this->replaceBySearch($string, $search);
        }

        return $changed;
    }

    /**
     * replaceBySearchList.
     *
     * @param string &$string Param
     * @param string &$search Param
     */
    private function replaceBySearch(&$string, &$search)
    {
        if (preg_match_all($search, $string, $matches, \PREG_SET_ORDER) < 1) {
            return false;
        }

        $changed = false;
        $isGenerateSrcsetEnabled = $this->params->get('generate_srcset');

        foreach ($matches as $match) {
            $fullMatch = $match['0'];
            $tagBegin = $match['1'];
            $fileQuery = trim($match['3']);
            $tagEnd = $match['4'];

            $file = $this->parseFileQuery($fileQuery);

            if (!$file) {
                continue;
            }

            if ($this->fileIsIgnored($file)) {
                continue;
            }

            // Step 1 - Adaptive Image
            $hasCurrentAdaptedImage = false;
            $hasAdaptedSrcSet = false;

            $imageProcessor = $this->getImageProcessor();
            $generateAdaptedImage = !$imageProcessor->isSvg($file) && !$this->imageFileIsIgnored($file);
            $step1File = $file;

            if ($generateAdaptedImage) {
                $step1File = $imageProcessor->getAdaptedImage($file);

                if (!$step1File) {
                    continue;
                }

                $hasCurrentAdaptedImage = false !== strpos($step1File, $this->aiCachePath);
                $hasAdaptedSrcSet = $isGenerateSrcsetEnabled && !empty($imageProcessor->getSrcSet());

                if ($hasCurrentAdaptedImage) {
                    $parts = explode($this->aiCachePath, $step1File);

                    // The new file
                    $file = $this->aiCachePath.$parts[1];
                }
            }

            if (!$hasCurrentAdaptedImage && !$this->hasCdn && !$hasAdaptedSrcSet) {
                // Nothing to do
                continue;
            }

            // Step 2 - Has AdaptedImage or Has CDN (Case SVG "Optimized" Image)
            $newSrcAttr = $tagBegin.$file.$tagEnd;
            $processedUrl = $file;
            $this->srcsetDictionary[$processedUrl] = $imageProcessor->getSrcSet();

            // Has CDN
            if ($this->hasCdn) {
                $processedUrl = $this->getMainCdnUrl().'/'.$file;
                $newSrcAttr = $tagBegin.$processedUrl.$tagEnd;
                $this->srcsetDictionary[$processedUrl] = $this->getCdnUrls($imageProcessor->getSrcSet());
            }

            // Replace Image - Adapted or Optimized
            $string = str_replace($fullMatch, $newSrcAttr, $string);
            $changed = true;
        }

        return $changed;
    }

    /**
     * setFiletypeSearches.
     */
    private function setFiletypeSearches()
    {
        if (empty($this->currentSet->filetypes)) {
            return;
        }

        $urls = $this->getUrlList();

        foreach ($urls as $url) {
            $this->setSearchesByUrl($url);
        }
    }

    /**
     * getUrlList.
     */
    private function getUrlList()
    {
        // Domain url or root path
        $url = preg_quote(str_replace('https://', 'http://', CMSUri::root()), '#');
        $url .= '|'.preg_quote(str_replace('http://', '//', CMSUri::root()), '#');

        if ($this->currentSet->enable_https) {
            $url .= '|'.preg_quote(str_replace('http://', 'https://', CMSUri::root()), '#');
        }

        if (CMSUri::root(1)) {
            $url .= '|'.preg_quote(CMSUri::root(1).'/', '#');
        }

        $filetypes = implode('|', $this->currentSet->filetypes);
        $root = preg_quote($this->currentSet->root, '#');

        $urls = [];

        // Absolute path
        $urls[] = '(?:'.$url.')'.$root.'([^ \?QUOTES]+\.(?:'.$filetypes.')(?:\?[^QUOTES]*)?)';

        // Relative path
        $urls[] = 'LSLASH'.$root.'([a-z0-9-_]+/[^ \?QUOTES]+\.(?:'.$filetypes.')(?:\?[^QUOTES]*)?)';

        // Relative path - file in root
        $urls[] = 'LSLASH'.$root.'([a-z0-9-_]+[^ \?\/QUOTES]+\.(?:'.$filetypes.')(?:\?[^QUOTES]*)?)';

        return $urls;
    }

    /**
     * setSearchesByUrl.
     *
     * @param string $url Param
     */
    private function setSearchesByUrl($url)
    {
        $urlRegex = '\s*'.str_replace('QUOTES', '"\'', $url).'\s*';

        if ($this->currentSet->enable_in_scripts) {
            $urlRegexJs = str_replace('LSLASH', '', $urlRegex);

            // "..."
            $this->currentSet->jsSearches[] = '#((["\']))'.$urlRegexJs.'(["\'])#i';
        }

        $urlRegex = str_replace('LSLASH', '/?', $urlRegex);

        // Attrib="..."
        $this->currentSet->searches[] = '#((?:'.$this->params->get('tag_attribs').')\s*(["\']))'.$urlRegex.'(\2)#i';

        // Attrib=...
        $this->currentSet->searches[] = '#((?:'.$this->params->get('tag_attribs').')())'.$urlRegex.'([\s|>])#i';

        // Url(...) or url("...")
        $this->currentSet->searches[] = '#(url\(\s*((?:["\'])?))'.$urlRegex.'(\2\s*\))#i';

        // Add ')' to the no quote checks
        $urlRegex = '\s*'.str_replace('QUOTES', '"\'\)', $url).'\s*';

        // Url(...)
        $this->currentSet->searches[] = '#(url\(\s*())'.$urlRegex.'(\s*\))#i';
    }

    /**
     * getMainCdnUrl.
     *
     * @return string
     */
    private function getMainCdnUrl()
    {
        $cdns = $this->currentSet->cdns;

        return $this->currentSet->protocol.$cdns['0'];
    }

    /**
     * getCdnUrls.
     *
     * @param string $file  param
     * @param mixed  $files
     *
     * @return string
     */
    private function getCdnUrls($files)
    {
        $processed = [];

        foreach ($files as $key => $file) {
            $processed[$key] = $this->getMainCdnUrl().'/'.$file;
        }

        return $processed;
    }

    /**
     * setCdnPaths.
     */
    private function setCdnPaths()
    {
        $this->currentSet->cdns = explode(',', $this->currentSet->cdn);

        foreach ($this->currentSet->cdns as $i => $cdn) {
            $cdn = preg_replace('#^.*\://#', '', trim($cdn));
            $this->currentSet->cdns[$i] = $cdn;
        }
    }

    /**
     * cleanLeftoverJunk - Just in case you can't figure the method name out: this cleans the left-over junk.
     *
     * @param string &$string Param
     */
    private function cleanLeftoverJunk(&$string)
    {
        $string = str_replace(['{nocdn}', '{/nocdn}'], '', $string);
    }

    /**
     * isEditPage - check if page is an edit page.
     *
     * @return bool
     */
    private function isEditPage()
    {
        $option = CMSFactory::getApplication()->input->get('option');

        // Always return false for these components
        if (in_array($option, ['com_rsevents', 'com_rseventspro'], true)) {
            return 0;
        }

        $task = CMSFactory::getApplication()->input->get('task');
        $view = CMSFactory::getApplication()->input->get('view');

        if (false !== strpos($task, '.')) {
            $task = explode('.', $task);
            $task = array_pop($task);
        }

        if (false !== strpos($view, '.')) {
            $view = explode('.', $view);
            $view = array_pop($view);
        }

        return
                in_array($task, ['edit', 'form', 'submission'], true)
                || in_array($view, ['edit', 'form'], true)
                || in_array(CMSFactory::getApplication()->input->get('do'), ['edit', 'form'], true)
                || in_array(CMSFactory::getApplication()->input->get('layout'), ['edit', 'form', 'write'], true)
                || in_array(CMSFactory::getApplication()->input->get('option'), ['com_contentsubmit', 'com_cckjseblod'], true)
                || $this->isAdmin()
        ;
    }

    private function getImageProcessor()
    {
        $imageProcessor = new ImageProcessor(
            JPATH_ROOT,
            $this->aiCachePath,
            $this->params->get('resolutions', '1382,992,768,480'),
            $this->params->get('jpg_quality', '85'),
            $this->params->get('sharpen', true),
            $this->params->get('convert_webp', false)
        );

        $imageProcessor->setGenerateSrcset($this->params->get('generate_srcset'));

        return $imageProcessor;
    }

    private function parseFileQuery($fileQuery)
    {
        $file = $fileQuery;

        // Joomla 4 Issue - [#35871] - Media field type now encondes image metadata into the actual value
        // https://issues.joomla.org/tracker/joomla-cms/35871
        // "images/1024px-Great_Wave_off_Kanagawa2-wb.jpg#joomlaImage://local-images/images/1024px-Great_Wave_off_Kanagawa2-wb.jpg?width=468&height=60"
        if (method_exists('Joomla\CMS\HTML\HTMLHelper', 'cleanImageURL')) {
            $imageUrl = \Joomla\CMS\HTML\HTMLHelper::cleanImageURL($fileQuery);
            $file = $imageUrl->url;
        }

        // "images/anuarios/2020/WhatsApp_Image_2020-12-17_at_102729.jpeg?5fdb673e"
        if (false !== strpos($file, '?')) {
            [$file, $restOfQuery] = explode('?', $file);
        }

        return $file;
    }
}
